-- name: create-users-table
--
-- The users of the DirectGPS system. The users in here, are also
-- added to the Postgres roles table as well, so we can use the
-- database for security.
CREATE TABLE users (
  user_name character varying(40) PRIMARY KEY,
  email character varying(255) NOT NULL,
  full_name character varying(200) NOT NULL,
  admin BOOLEAN NOT NULL DEFAULT false,
  password character varying(100),
  upload_gps_only BOOLEAN NOT NULL DEFAULT false,
  email_verified BOOLEAN NOT NULL DEFAULT false,
  inactive BOOLEAN NOT NULL DEFAULT false
);

-- name: ensure-user-names-unique
--
-- Create an index here, just so we won't get any duplicate user
-- names.
CREATE UNIQUE INDEX users_sanatise_user_name ON users (
  REMOVE_WHITESPACE(LOWER(user_name))
);

CREATE UNIQUE INDEX users_sanatise_user_email ON users (
  REMOVE_WHITESPACE(LOWER(email))
);

-- name: enable-row-level-security
ALTER TABLE users ENABLE ROW LEVEL SECURITY;

-- name: create-is-admin-function
--
-- Return true if the given user is an administrator.
CREATE OR REPLACE FUNCTION is_admin(inUsername TEXT)
RETURNS BOOLEAN AS
$BODY$
DECLARE
  result BOOLEAN;
BEGIN
  SELECT true into result FROM users WHERE $1 = user_name AND admin IS true;
  RETURN result;
END
$BODY$
LANGUAGE 'plpgsql' IMMUTABLE
SECURITY DEFINER
SET search_path = public, pg_temp;

-- name: create-find-login-name
CREATE OR REPLACE FUNCTION find_login(inEmail TEXT)
RETURNS TEXT AS
$BODY$
DECLARE
  result TEXT;
BEGIN
  SELECT user_name INTO result FROM users WHERE LOWER($1) LIKE email;
  RETURN result;
END
$BODY$
LANGUAGE 'plpgsql' IMMUTABLE
SECURITY DEFINER
SET search_path = public, pg_temp;

-- name: revoke-find-login
--- Only the directgps_login_query
REVOKE ALL ON FUNCTION find_login(inEmail TEXT) FROM public;
GRANT EXECUTE ON FUNCTION find_login(inEmail TEXT) TO directgps_login_query;

-- name: grant-select-public
GRANT SELECT (email, full_name, user_name, admin) ON users TO PUBLIC;

-- name: grant-update-directgps-user
GRANT UPDATE (email, full_name, password) ON users TO directgps_user;

-- name: grant-all-to-directgps-admin
--
-- Admins can INSERT and UPDATE.
GRANT SELECT,UPDATE,INSERT,DELETE ON users TO directgps_admin;

-- name: create-directgps-user-select-policy
-- Users can read their own data.
-- Admins can read everybody's.
CREATE POLICY user_read_own_data ON users
  FOR SELECT USING (current_user = user_name OR
                    (SELECT is_admin(current_user)));

-- name: create-directgps-user-update-policy
--
-- Users can modify their own data, but not change the username.
CREATE POLICY user_modify_own_data ON users
  FOR UPDATE USING (current_user = user_name)
  WITH CHECK (user_name = current_user);

-- name: ensure-user-fields-sanitised-function
CREATE OR REPLACE FUNCTION users_sanitise_whitespace()
  RETURNS trigger AS
$BODY$
BEGIN
  NEW.email := REMOVE_WHITESPACE(NEW.email);
  NEW.full_name := SANITISE_WHITESPACE(NEW.full_name);
  RETURN NEW;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE;

-- name: create-user-fields-sanatise-trigger
CREATE TRIGGER sanatise_user_fields
  BEFORE INSERT OR UPDATE OF email,full_name ON users
  FOR EACH ROW
  EXECUTE PROCEDURE users_sanitise_whitespace();

-- name: create-always-admin-function
--
-- When a user is updated, or deleted, make sure there's always going
-- to be an admin present on the system.
CREATE OR REPLACE FUNCTION admin_must_remain()
  RETURNS trigger AS
$BODY$
DECLARE
  admins_left int = 0;
  error_occured boolean = false;
  error_msg text = 'user,9006,';
  result RECORD;
BEGIN
  IF TG_OP = 'UPDATE' THEN
    result := NEW;
  ELSE
    result := OLD;
  END IF;

  IF OLD.user_name = current_user AND NEW.admin = false THEN
    error_msg := error_msg || 'Cannot remove own admin access;';
    error_occured := true;
  ELSE
    SELECT COUNT(*) INTO admins_left
    FROM users
    WHERE (user_name <> OLD.user_name
           AND admin = true);

    IF admins_left = 0 THEN
      IF TG_OP = 'UPDATE' THEN
        IF NEW.admin = false THEN
          error_msg := error_msg || 'Need at least one admin;';
          error_occured := true;
        ELSE
          result := NEW;
        END IF;
      ELSE
        error_msg := error_msg || 'Must not delete last admin;';
        error_occured := true;
      END IF;
    END IF;

    IF error_occured THEN
      RAISE EXCEPTION '%',error_msg;
      RETURN NULL;
    END IF;
  END IF;

  RETURN result;
END;
$BODY$
LANGUAGE 'plpgsql' IMMUTABLE;

--- Triggers so we will always have an admin in the database.
-- name: create-last-admin-update-trigger
CREATE TRIGGER maintain_last_admin_update
  BEFORE UPDATE OF admin ON users
  FOR EACH ROW
  EXECUTE PROCEDURE admin_must_remain();

-- name: create-last-admin-delete-trigger
CREATE TRIGGER maintain_last_admin_delete
  BEFORE DELETE ON users
  FOR EACH ROW
  EXECUTE PROCEDURE admin_must_remain();

-- name: create-remove-directgps-user-function
--
-- After a user is deleted, remove them from the Postgresql user list
-- too.
CREATE OR REPLACE FUNCTION remove_directgps_user_from_postgresql()
  RETURNS trigger AS
$BODY$
BEGIN
  EXECUTE 'DROP ROLE ' || quote_ident(OLD.user_name);
  RETURN OLD;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE;

-- name: create-remove-directgps-user-trigger
CREATE TRIGGER remove_directgps_user_from_postgresql_trigger
  AFTER DELETE ON users
  FOR EACH ROW
  EXECUTE PROCEDURE remove_directgps_user_from_postgresql();

-- name: create-update-user-directgps-role-function
--
-- Depending on user's admin status, allocate them to the matching
-- postgresql role.
CREATE OR REPLACE FUNCTION update_admin_to_role()
  RETURNS trigger AS
$BODY$
BEGIN
  -- Catch username changes.
  IF OLD.user_name <> NEW.user_name THEN
    EXECUTE 'ALTER ROLE ' || quote_ident(OLD.user_name) || ' RENAME TO ' ||
      quote_ident(NEW.user_name);
  END IF;

  IF NEW.admin <> OLD.admin THEN
    IF NEW.admin = true THEN
      EXECUTE 'GRANT directgps_admin TO ' || quote_ident(NEW.user_name);
    ELSIF NEW.admin <> OLD.admin THEN
      EXECUTE 'REVOKE directgps_admin FROM ' || quote_ident(NEW.user_name);
    END IF;
  END IF;

  RETURN NEW;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE;

-- name: create-update-user-directgps-role-trigger
CREATE TRIGGER update_admin_to_role_trigger
  AFTER UPDATE OF user_name, admin ON users
  FOR EACH ROW
  EXECUTE PROCEDURE update_admin_to_role();

-- name: create-add-user-postgres-function
--
-- This trigger is fired when a user is inserted, it will add the user
-- to the actual database user list.
CREATE OR REPLACE FUNCTION add_directgps_user_to_postgres()
  RETURNS TRIGGER AS
$BODY$
BEGIN
  -- If the username is Null, we won't add.
  IF ((NEW.user_name IS NOT NULL) AND (NEW.password IS NOT NULL)) THEN
    EXECUTE 'CREATE USER ' || quote_ident(NEW.user_name) || ' WITH PASSWORD ' ||
      quote_literal(NEW.password);

    IF NEW.admin THEN
      EXECUTE 'GRANT directgps_admin TO ' || quote_ident(NEW.user_name);
    ELSE
      EXECUTE 'GRANT directgps_user TO ' || quote_ident(NEW.user_name);
    END IF;
  END IF;
  --- Never store plaintext password.
  NEW.password := NULL;

  RETURN NEW;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE;

-- name: create-add-user-postgres-trigger
--
-- Must be fired before, otherwise password is not blanked.
CREATE TRIGGER adding_user_to_directgps
  BEFORE INSERT ON users
  FOR EACH ROW
  EXECUTE PROCEDURE add_directgps_user_to_postgres();

-- name: create-change-user-postgres-password-function
--
-- Change users password
--
-- UPDATE trigger when a user's password is changed, to keep the
-- PostgreSQL user list in sync.
--
-- TODO: This will result in a mismatch if we have failed constraint
-- checks?
CREATE OR REPLACE FUNCTION update_password()
  RETURNS TRIGGER AS
$BODY$
BEGIN
  --- Password should be null anyway, since it always gets blanked.
  IF NEW.password IS NOT NULL THEN
    EXECUTE 'ALTER ROLE ' || quote_ident(OLD.user_name) ||
      ' WITH PASSWORD ' || quote_literal(NEW.password);
  END IF;

  -- Always blank the password, even if it's not changed.
  NEW.password := NULL;

  RETURN NEW;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE;

-- name: create-change-user-postgres-password-trigger
CREATE TRIGGER update_password_trigger
  BEFORE UPDATE OF password ON users
  FOR EACH ROW
  EXECUTE PROCEDURE update_password();

-- name: create-begin-email-verify-function
--
-- This is fired when a new user is created, or, if the e-mail
-- address has been changed. It will insert into an e-mail
-- verification table, which has a trigger in it that will send off
-- an e-mail request.
CREATE OR REPLACE FUNCTION begin_email_verification()
  RETURNS TRIGGER AS
$BODY$
BEGIN
  IF TG_OP = 'UPDATE' THEN
    IF (OLD.email <> NEW.email) THEN
      INSERT INTO emails_to_verify (email) VALUES (NEW.email);
    END IF;
  ELSE
    INSERT INTO emails_to_verify (email) VALUES (NEW.email);
  END IF;

  RETURN NEW;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE;

-- name: create-begin-email-verify-trigger
CREATE TRIGGER begin_email_verification_trigger
  AFTER INSERT ON users
  FOR EACH ROW
  EXECUTE PROCEDURE begin_email_verification();

-- name: create-email-verify-table
--
-- This is an insert only table, a user will insert into this table
-- whenever their e-mail address is changed. An insert, will fire a
-- trigger that will then send off an e-mail and add the expected
-- response to the e-mail verification table so we can verify their
-- response.
CREATE TABLE emails_to_verify (
  request_number integer,
  email character varying(255),
  md5 character(32),
  request_timestamp TIMESTAMP WITHOUT TIME ZONE
);

-- name: revoke-all-on-verify-table
REVOKE ALL ON TABLE emails_to_verify FROM PUBLIC;

-- name: grant-insert-on-verify-table
GRANT INSERT ON TABLE emails_to_verify TO PUBLIC;

-- name: create-send-email-verification-function
--
-- This function will do a basic check, then send off an e-mail so we
-- can verify the e-mail given the user's response.
CREATE OR REPLACE FUNCTION send_email_verification()
  RETURNS TRIGGER AS
$BODY$
DECLARE
  user_valid int;
  email_content varchar;
  smtp_server varchar;
  from_address varchar;
  site_url varchar;
BEGIN
  -- Only execute if the user's id is valid.
  SELECT INTO user_valid Count(*)
    FROM users WHERE email = NEW.email;

  IF user_valid > 0 THEN
    SELECT INTO email_content variable_value
      FROM system_config
     WHERE variable_name = 'EMAIL_VERIFY_CONTENT';
    SELECT INTO smtp_server variable_value
      FROM system_config
     WHERE variable_name = 'SMTP_SERVER';
    SELECT INTO from_address variable_value
      FROM system_config
     WHERE variable_name = 'EMAIL_FROM';
    SELECT INTO site_url variable_value
      FROM system_config
     WHERE variable_name = 'SITE_URL';

    IF from_address IS NULL THEN from_address = ''; END IF;

    NEW.md5 := md5(now()::varchar);

    --- Send email here
    --- PERFORM send_email_function(smtp_server,
    ---                            NEW.email,
    ---                            from_address,
    ---                            'DirectGPS - Email verification',
    ---                            email_content || site_url || NEW.md5);
    RETURN NEW;
  END IF;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE
SECURITY DEFINER
SET search_path = public, pg_temp;

-- name: create-send-mail-verification-trigger
--
-- This trigger causes a verification e-mail to be sent to the user.
CREATE TRIGGER send_email_verification_trigger
  BEFORE INSERT ON emails_to_verify
  FOR EACH ROW
  EXECUTE PROCEDURE send_email_verification();

-- name: create-email-confirmation-table
--
-- Table to be inserted into so users can verify their e-mail address.
CREATE TABLE emails_confirmed (
  confirmation_number integer,
  email character varying(255),
  md5 character(32)
);

-- name: revoke-all-on-confirm-table
REVOKE ALL ON TABLE emails_confirmed FROM PUBLIC;

-- name: grant-insert-on-confirm-table
GRANT INSERT ON TABLE emails_confirmed TO PUBLIC;

-- name: create-confirm-email-function
--
-- Function to be called if the user inserts to confirm their e-mail
-- address.
CREATE OR REPLACE FUNCTION receive_email_verification_function()
  RETURNS TRIGGER AS
$BODY$
DECLARE
  request_valid int;
  error_count int = 0;
  error_list text = '';
BEGIN
  -- Read the first table to see if the link is valid and has not
  -- expired.
  SELECT INTO request_valid COUNT(*)
    FROM emails_to_verify
    WHERE email = NEW.email
      AND md5 = NEW.md5;

  IF request_valid = 0 THEN
    error_count = error_count + 1;
    error_list = error_list || 'email,9005,Invalid Link;';
  ELSE
    UPDATE users SET email_verified = TRUE
      WHERE email = NEW.email;
  END IF;

  IF error_count > 0 THEN
    RAISE EXCEPTION '%',error_list;
    RETURN NULL;
  ELSE
    RETURN NEW;
  END IF;
END;
$BODY$
LANGUAGE 'plpgsql' VOLATILE
SECURITY DEFINER
SET search_path = public, pg_temp;

-- name: Trigger when e-mail verification is inserted.
CREATE TRIGGER receive_email_verification_trigger
  AFTER INSERT ON emails_confirmed
  FOR EACH ROW
  EXECUTE PROCEDURE receive_email_verification_function();
